/**
 * SERVIÇO DE API PARA CHAT - VERSÃO OTIMIZADA E VALIDADA
 * Compatível com Concord CRM (Innoclapps) - fallback para fetch
 * Foco em performance, robustez, clareza e manutenibilidade.
 */

let API_CONFIG = {
  baseURL: '/api/chat', // ✅ CORRIGIDO: Caminho base específico do chat
  timeout: 30000,
  retries: 3,
  retryDelay: 1000,
  cacheTTL: 300000,
  debug: false
}

/**
 * Cache local para requisições GET, utilizando Map para melhor performance.
 * @type {Map<string, {data: any, timestamp: number}>}
 */
const cache = new Map()

/**
 * Função para debug condicional.
 * @param {...any} args Argumentos a serem logados.
 */
function debug(...args) {
  if (API_CONFIG.debug) console.log('🌐 API:', ...args)
}

/**
 * Normaliza os parâmetros de uma URL para uso consistente em chaves de cache.
 * @param {object} params Parâmetros da URL.
 * @returns {object} Parâmetros normalizados.
 */
function normalizeParams(params = {}) {
  const normalized = {}
  if (!params || typeof params !== 'object') return normalized
  Object.keys(params).sort().forEach(k => {
    const v = params[k]
    if (v !== null && v !== undefined && v !== '') {
      normalized[k] = String(v).trim()
    }
  })
  return normalized
}

/**
 * Gera uma chave de cache única para uma requisição.
 * @param {string} method Método HTTP.
 * @param {string} endpoint Endpoint da API.
 * @param {object} params Parâmetros da requisição.
 * @returns {string} A chave de cache.
 */
function getCacheKey(method, endpoint, params = {}) {
  const normalized = normalizeParams(params)
  const paramsString = Object.keys(normalized).length ? JSON.stringify(normalized) : ''
  return `${method.toLowerCase()}:${endpoint.toLowerCase()}:${paramsString}`
}

/**
 * Verifica se um item no cache ainda é válido.
 * @param {number} timestamp Timestamp de criação do item no cache.
 * @returns {boolean} True se o item é válido.
 */
function isValidCache(timestamp) {
  return Date.now() - timestamp < API_CONFIG.cacheTTL
}

/**
 * Tenta recuperar dados do cache.
 * @param {string} key Chave do cache.
 * @returns {any|null} Os dados cacheados ou null.
 */
function getFromCache(key) {
  const item = cache.get(key)
  if (!item) return null
  if (isValidCache(item.timestamp)) {
    debug('📦 Cache hit:', key)
    return item.data
  }
  cache.delete(key)
  debug('🗑️ Cache expirado removido:', key)
  return null
}

/**
 * Armazena dados no cache.
 * @param {string} key Chave do cache.
 * @param {any} data Dados a serem armazenados.
 */
function setCache(key, data) {
  cache.set(key, { data, timestamp: Date.now() })
  debug('💾 Dados cacheados:', key)
}

/**
 * Limpa entradas do cache que correspondem a um padrão.
 * @param {string|null} pattern Padrão para limpeza ou null para limpar tudo.
 */
function clearCache(pattern = null) {
  if (!pattern) {
    cache.clear()
    debug('🧹 Cache totalmente limpo')
    return
  }
  const lowerPattern = pattern.toLowerCase()
  const keysToDelete = []
  for (const k of cache.keys()) {
    if (k.includes(lowerPattern)) {
      keysToDelete.push(k)
    }
  }
  keysToDelete.forEach(k => cache.delete(k))
  debug(`🧹 Cache limpo para padrão "${pattern}": ${keysToDelete.length} entradas removidas`)
}

/**
 * ✅ CORRIGIDO: Obtém o token CSRF do HTML ou configuração global.
 * @returns {string} Token CSRF.
 */
function getCsrfToken() {
  try {
    // ✅ PRIMEIRA TENTATIVA: Innoclapps scriptConfig
    if (window.Innoclapps?.scriptConfig) {
      const token = window.Innoclapps.scriptConfig('chat')?.csrf_token
      if (token) return token
    }
    
    // ✅ SEGUNDA TENTATIVA: chatConfig global
    if (window.chatConfig?.csrf_token) return window.chatConfig.csrf_token
    
    // ✅ TERCEIRA TENTATIVA: Meta tag
    const meta = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
    if (meta) return meta
    
    // ✅ QUARTA TENTATIVA: Laravel global (se existir)
    if (window.Laravel?.csrfToken) return window.Laravel.csrfToken
    
  } catch (e) {
    debug('❌ Erro ao obter CSRF token:', e)
  }
  return ''
}

/**
 * Retorna os headers padrão para requisições API.
 * @returns {object} Headers padrão.
 */
function getDefaultHeaders() {
  const headers = {
    'Accept': 'application/json',
    'X-Requested-With': 'XMLHttpRequest'
  }
  const token = getCsrfToken()
  if (token) headers['X-CSRF-TOKEN'] = token
  return headers
}

/**
 * ✅ CORRIGIDO: Constrói uma URL completa a partir de um endpoint.
 * @param {string} endpoint O endpoint da API.
 * @param {object} params Parâmetros de query.
 * @returns {string} A URL completa.
 */
function buildUrl(endpoint, params = {}) {
  try {
    let base = API_CONFIG.baseURL
    base = base.endsWith('/') ? base.slice(0, -1) : base

    let path = endpoint.startsWith('/') ? endpoint : `/${endpoint}`
    let fullPath = base + path

    const url = new URL(fullPath, window.location.origin)

    if (params && typeof params === 'object') {
      Object.keys(params).forEach(key => {
        const v = params[key]
        if (v !== null && v !== undefined && v !== '') {
          url.searchParams.append(key, String(v))
        }
      })
    }
    return url.toString()
  } catch (e) {
    debug('❌ Erro ao construir URL:', e, { endpoint, params, baseURL: API_CONFIG.baseURL })
    return `${API_CONFIG.baseURL}${endpoint.startsWith('/') ? '' : '/'}${endpoint}`
  }
}

/**
 * Tenta obter o cliente HTTP (Axios via Innoclapps ou fetch).
 * @returns {{type: 'axios'|'fetch', instance: any|null}}
 */
function getHttpClient() {
  if (window.Innoclapps?.request) {
    return { type: 'axios', instance: window.Innoclapps.request() }
  }
  return { type: 'fetch', instance: null }
}

/**
 * ✅ CORRIGIDO: Realiza uma requisição HTTP genérica.
 * @param {string} method Método HTTP.
 * @param {string} endpoint Endpoint da API.
 * @param {object|FormData|null} payload Dados da requisição.
 * @param {object} options Opções adicionais.
 * @returns {Promise<any>} A resposta da API.
 */
async function makeRequest(method, endpoint, payload = null, options = {}) {
  const start = Date.now()
  let lastError = null
  const client = getHttpClient()

  const queryParams = (method === 'GET' && payload && typeof payload === 'object') ? payload : {}
  const requestBody = (method !== 'GET') ? payload : undefined
  const requestUrl = buildUrl(endpoint, queryParams)

  for (let attempt = 1; attempt <= API_CONFIG.retries; attempt++) {
    try {
      debug(`📤 ${method} ${requestUrl} (tentativa ${attempt})`)

      // Verifica cache para GET
      if (method === 'GET' && options.cache !== false) {
        const ck = getCacheKey(method, endpoint, queryParams)
        const cached = getFromCache(ck)
        if (cached) return cached
      }

      // === CLIENTE AXIOS ===
      if (client.type === 'axios') {
        const axiosOpts = {
          method,
          url: requestUrl,
          headers: { ...getDefaultHeaders(), ...(options.headers || {}) },
          timeout: API_CONFIG.timeout,
          data: requestBody,
          responseType: options.responseType || 'json'
        }
        const resp = await client.instance.request(axiosOpts)
        const duration = Date.now() - start
        debug(`📥 Axios ${requestUrl} (${resp.status}) - ${duration}ms`)

        if (resp.status < 200 || resp.status >= 300) {
          const err = new Error(resp.data?.message || `HTTP ${resp.status}`)
          err.status = resp.status
          err.data = resp.data
          throw err
        }

        const respData = resp.data
        if (method === 'GET' && options.cache !== false) {
          setCache(getCacheKey(method, endpoint, queryParams), respData)
        }
        if (['POST', 'PUT', 'PATCH', 'DELETE'].includes(method)) {
          postMutationCleanup(endpoint)
        }
        return respData
      }

      // === FALLBACK: FETCH ===
      const fetchOpts = {
        method,
        headers: { ...getDefaultHeaders(), ...(options.headers || {}) },
        signal: AbortSignal.timeout(API_CONFIG.timeout)
      }

      if (method !== 'GET' && requestBody) {
        if (requestBody instanceof FormData) {
          fetchOpts.body = requestBody
        } else {
          fetchOpts.headers['Content-Type'] = 'application/json'
          fetchOpts.body = JSON.stringify(requestBody)
        }
      }

      const resp = await fetch(requestUrl, fetchOpts)
      const duration = Date.now() - start
      debug(`📥 Fetch ${requestUrl} (${resp.status}) - ${duration}ms`)

      const contentType = resp.headers.get('content-type') || ''
      let respData = null
      if (contentType.includes('application/json')) {
        try {
          respData = await resp.json()
        } catch (jsonError) {
          respData = await resp.text()
          debug('⚠️ Falha ao parsear JSON, fallback para texto:', respData, jsonError)
        }
      } else {
        respData = await resp.text()
      }

      if (!resp.ok) {
        let errorMessage = `HTTP ${resp.status}`

        if (respData && typeof respData === 'object' && (respData.message || respData.error)) {
          errorMessage = respData.message || respData.error
        } else if (typeof respData === 'string' && respData) {
          errorMessage = respData
        }

        if (resp.status === 401) errorMessage = 'Não autorizado - faça login novamente.'
        else if (resp.status === 403) errorMessage = 'Acesso negado.'
        else if (resp.status === 404) errorMessage = 'Recurso não encontrado.'
        else if (resp.status === 422) errorMessage = 'Dados inválidos. Verifique as informações fornecidas.'
        else if (resp.status >= 500) errorMessage = 'Erro interno do servidor. Tente novamente mais tarde.'

        const err = new Error(errorMessage)
        err.status = resp.status
        err.data = respData
        throw err
      }

      if (method === 'GET' && options.cache !== false) {
        setCache(getCacheKey(method, endpoint, queryParams), respData)
      }
      if (['POST', 'PUT', 'PATCH', 'DELETE'].includes(method)) {
        postMutationCleanup(endpoint)
      }
      return respData

    } catch (error) {
      lastError = error
      const errorStatus = error.status || error.response?.status
      debug(`❌ ${method} ${requestUrl} - tentativa ${attempt} falhou:`, error.message || error, errorStatus ? `(Status: ${errorStatus})` : '')
      
      if (errorStatus >= 400 && errorStatus < 500 && errorStatus !== 408 && errorStatus !== 429 && errorStatus !== 422) {
        throw error
      }
      if (attempt < API_CONFIG.retries) {
        const backoff = API_CONFIG.retryDelay * Math.pow(2, attempt - 1)
        await new Promise(r => setTimeout(r, backoff))
      }
    }
  }

  debug(`❌ ${method} ${requestUrl} - todas as tentativas falharam`)
  throw lastError || new Error('Falha na requisição após todas as tentativas.')
}

/**
 * Limpeza de cache após mutação.
 * @param {string} endpoint O endpoint modificado.
 */
function postMutationCleanup(endpoint) {
  try {
    clearCache('/conversations')
    const match = endpoint.match(/\/conversations\/(\d+)/)
    if (match?.[1]) {
      const id = match[1]
      clearCache(`/conversations/${id}`)
      clearCache(`/conversations/${id}/messages`)
    }
  } catch (e) {
    debug('❌ Erro no pós-limpeza de cache:', e)
  }
}

/**
 * ✅ OBJETO DE SERVIÇO API CORRIGIDO
 */
const apiService = {
  // ✅ PROPRIEDADES PARA COMPATIBILIDADE
  baseUrl: API_CONFIG.baseURL,
  
  // ✅ MÉTODO CSRF CORRIGIDO
  getCsrfToken() {
    return getCsrfToken()
  },

  // === Métodos HTTP ===
  async get(url, params = null, options = {}) { 
    return makeRequest('GET', url, params, options) 
  },
  async post(url, data = null, options = {}) { 
    return makeRequest('POST', url, data, options) 
  },
  async put(url, data = null, options = {}) { 
    return makeRequest('PUT', url, data, options) 
  },
  async patch(url, data = null, options = {}) { 
    return makeRequest('PATCH', url, data, options) 
  },
  async delete(url, options = {}) { 
    return makeRequest('DELETE', url, null, options) 
  },

  // === Configuração ===
  setBaseURL(url) { 
    API_CONFIG.baseURL = url
    this.baseUrl = url // ✅ MANTER COMPATIBILIDADE
    debug('🔧 Base URL configurada:', url) 
  },
  setTimeout(ms) { 
    if (typeof ms === 'number' && ms > 0) API_CONFIG.timeout = ms 
  },
  setCacheTTL(ms) { 
    if (typeof ms === 'number' && ms > 0) API_CONFIG.cacheTTL = ms 
  },
  setDebug(enabled) { 
    API_CONFIG.debug = !!enabled
    debug('🐛 Debug', API_CONFIG.debug) 
  },
  getCacheStats() { 
    return { 
      size: cache.size, 
      keys: Array.from(cache.keys()), 
      config: {...API_CONFIG} 
    } 
  },
  clearCache(pattern = null) { 
    clearCache(pattern) 
  },

  // === ✅ MÉTODOS ESPECÍFICOS CORRIGIDOS ===

  /**
   * ✅ OBTER CONVERSAS COM PAGINAÇÃO
   */
  async getConversations(params = {}) {
    try {
      const queryParams = new URLSearchParams()
      
      if (params.page) queryParams.append('page', params.page)
      if (params.per_page) queryParams.append('per_page', params.per_page)
      if (params.search) queryParams.append('search', params.search)
      if (params.status && params.status !== 'todas') queryParams.append('status', params.status)
      if (params.type) queryParams.append('type', params.type)
      if (params.assigned) queryParams.append('assigned', params.assigned)
      if (params.only_unread) queryParams.append('only_unread', 'true')
      if (params.show_all) queryParams.append('show_all', 'true')
      if (params.include_archived) queryParams.append('include_archived', 'true')

      const endpoint = `/conversations${queryParams.toString() ? '?' + queryParams.toString() : ''}`
      return await this.get(endpoint)
    } catch (error) {
      console.error('Erro ao buscar conversas:', error)
      throw error
    }
  },

  /**
   * ✅ MARCAR COMO LIDA
   */
  async markAsRead(conversationId) {
    try {
      return await this.post(`/conversations/${conversationId}/mark-as-read`)
    } catch (error) {
      console.error('Erro ao marcar como lida:', error)
      throw error
    }
  },

  /**
   * ✅ FIXAR/DESAFIXAR CONVERSA
   */
  async togglePinConversation(conversationId, isPinned) {
    try {
      return await this.patch(`/conversations/${conversationId}`, {
        is_pinned: isPinned
      })
    } catch (error) {
      console.error('Erro ao alterar fixação:', error)
      throw error
    }
  },

  /**
   * ✅ SILENCIAR/ATIVAR NOTIFICAÇÕES
   */
  async toggleMuteConversation(conversationId, isMuted) {
    try {
      return await this.patch(`/conversations/${conversationId}`, {
        is_muted: isMuted
      })
    } catch (error) {
      console.error('Erro ao alterar silenciamento:', error)
      throw error
    }
  },

  /**
   * ✅ ALTERAR STATUS DA CONVERSA
   */
  async toggleConversationStatus(conversationId, status, reason = null) {
    try {
      return await this.post(`/conversations/${conversationId}/status`, {
        status: status,
        reason: reason
      })
    } catch (error) {
      console.error('Erro ao alterar status:', error)
      throw error
    }
  },

  /**
   * ✅ EXPORTAR CONVERSA
   */
  async exportConversation(conversationId, format = 'pdf') {
    try {
      return await this.post(`/conversations/${conversationId}/export`, {
        format: format
      })
    } catch (error) {
      console.error('Erro ao exportar conversa:', error)
      throw error
    }
  },

  /**
   * ✅ EXCLUIR CONVERSA
   */
  async deleteConversation(conversationId) {
    try {
      return await this.delete(`/conversations/${conversationId}`)
    } catch (error) {
      console.error('Erro ao excluir conversa:', error)
      throw error
    }
  },

  /**
   * ✅ OBTER USUÁRIOS
   */
  async getUsers() {
    try {
      return await this.get('/users')
    } catch (error) {
      console.error('Erro ao buscar usuários:', error)
      throw error
    }
  },

  /**
   * ✅ ATRIBUIR CONVERSA
   */
  async assignConversation(conversationId, userId) {
    try {
      return await this.post(`/conversations/${conversationId}/assign`, {
        user_id: userId
      })
    } catch (error) {
      console.error('Erro ao atribuir conversa:', error)
      throw error
    }
  },

  /**
   * ✅ OBTER ESTATÍSTICAS
   */
  async getStats() {
    try {
      return await this.get('/stats')
    } catch (error) {
      console.error('Erro ao buscar estatísticas:', error)
      throw error
    }
  },

  /**
   * ✅ CRIAR CONVERSA EXTERNA
   */
  async createExternalConversation(data) {
    try {
      return await this.post('/conversations', data)
    } catch (error) {
      console.error('Erro ao criar conversa externa:', error)
      throw error
    }
  },

  /**
   * ✅ CRIAR CHAT INTERNO
   */
  async createInternalChat(data) {
    try {
      return await this.post('/conversations/internal', data)
    } catch (error) {
      console.error('Erro ao criar chat interno:', error)
      throw error
    }
  },

  // === Outros métodos mantidos ===
  async getContacts(params = {}) {
    if (!params.search?.trim()) return this.getRecentContacts(params.limit || 50)
    const p = { q: params.search.trim(), limit: params.limit || 20 }
    const data = await this.get('/contacts/search', p)
    return Array.isArray(data) ? data : data?.data || []
  },

  async getRecentContacts(limit = 10) {
    const data = await this.get('/contacts/recent', { limit })
    return Array.isArray(data) ? data : data?.data || []
  },

  async createContact(payload) { 
    return this.post('/contacts', payload) 
  },

  async getAvailableEvolutionInstances() {
    try {
      console.log('📋 Buscando instâncias Evolution...')
      const response = await this.get('/evolution-instances')
      console.log('✅ Resposta da API:', response)
      
      if (!response || typeof response !== 'object') {
        throw new Error('Resposta inválida da API')
      }
      
      return {
        local_instances: response.local_instances || [],
        evolution_instances: response.evolution_instances || [],
        summary: response.summary || {}
      }
    } catch (error) {
      console.error('❌ Erro ao buscar instâncias:', error)
      throw error
    }
  },

  async getMessages(conversationId, params = {}) {
    const query = { page: params.page || 1, per_page: params.per_page || 50 }
    if (params.before_id) query.before_id = params.before_id
    const resp = await this.get(`/conversations/${conversationId}/messages`, query)
    
    if (!resp) return { messages: [], pagination: { current_page: 1, per_page: query.per_page, has_more: false, total: 0 } }
    if (Array.isArray(resp)) {
      return {
        messages: resp,
        pagination: { current_page: query.page, per_page: query.per_page, has_more: resp.length === query.per_page, total: resp.length }
      }
    }
    if (Array.isArray(resp.data)) {
      return {
        messages: resp.data,
        pagination: resp.pagination || { current_page: query.page, per_page: query.per_page, has_more: false, total: resp.data.length }
      }
    }
    return { messages: [], pagination: { current_page: 1, per_page: query.per_page, has_more: false, total: 0 } }
  },

  async sendMessage(conversationId, messageData) {
    let payload = {}
    if (typeof messageData === 'string') {
      payload = { content: messageData.trim(), message_type: 'text', timestamp: new Date().toISOString() }
    } else if (messageData && typeof messageData === 'object') {
      payload = {
        content: (messageData.content || messageData.message || messageData.text || '').trim(),
        message_type: messageData.type || messageData.message_type || 'text',
        timestamp: messageData.timestamp || new Date().toISOString(),
        attachments: messageData.attachments,
        reply_to: messageData.reply_to,
        metadata: messageData.metadata
      }
    }
    if (!payload.content) throw new Error('Conteúdo da mensagem é obrigatório.')
    return this.post(`/conversations/${conversationId}/messages`, payload)
  },

  async sendMessageWithAttachments(conversationId, formData) {
    if (!(formData instanceof FormData)) {
      throw new Error('Dados devem ser FormData para envio de anexos.')
    }
    const endpoint = `/conversations/${conversationId}/messages`
    return makeRequest('POST', endpoint, formData)
  },

  async sendTypingIndicator(conversationId, isTyping = true) { 
    return this.post(`/conversations/${conversationId}/typing`, { 
      is_typing: isTyping, 
      timestamp: new Date().toISOString() 
    }) 
  },

  async getConversationStats() { 
    return this.get('/stats') 
  },

  // Helpers
  invalidateMessagesCache(conversationId) {
    clearCache(`/conversations/${conversationId}/messages`)
    debug(`🧹 Cache de mensagens invalidado para conversa: ${conversationId}`)
  },

  invalidateConversationsCache() {
    clearCache('/conversations')
    clearCache('/conversations/archived')
    clearCache('/conversations/summary')
    debug('🧹 Cache de conversas invalidado')
  },

  buildUrl(endpoint, params = {}) { 
    return buildUrl(endpoint, params) 
  },

  getHeaders() { 
    return getDefaultHeaders() 
  },

  async healthCheck() { 
    return this.get('/chat-test') 
  }
}

debug('🚀 Serviço de API inicializado', { 
  baseURL: API_CONFIG.baseURL, 
  debug: API_CONFIG.debug, 
  cacheTTL: API_CONFIG.cacheTTL 
})

export default apiService